function chris_osc_fit()
% CHRIS_OSC_FIT()
%
% New function to analyze oscillation data.
% Reads from Excel file (edit the code to change the file).
% Loosely based on old function chris_osc.

% May 15 2014: Created.
% May 19 2014: Troubleshooted to produce meaningful results for frequencies data.
% May 23 2014: Extended to work with amplitudes data.
% Aug 15 2014: A change in how output is plotted (less dense format now).

mode = 1;       % 1 for frequencies, 2 for amplitudes

doGlobal = 1;                       % Spiking as a funciton of bumpGroup (highest-level analysis)
doInactivation = 0;                 % Rapid inactivation of high-frequency group (first group) for FREQ protocol.
oneGiantPlot = 1;                   % Whether we need one giant plot

list = [22003];               % If set, restricts analysis to this subset of cells


fileName = 'C:\Users\Arseny\Documents\4_Cells Classification\2014 data\BumpsData.xlsx';
if(mode==1)
    temp = xlsread(fileName,'RawFreq');
else
    temp = xlsread(fileName,'RawAmps');
end

% temp = temp(1:8,:);     %%% ----------- For DEBUGGING. Comment if not needed.
% temp = temp(end-23:end,:);
% temp = temp(sort([2 3 19 34 20 53 74 166 12 13 111 92 142 76 97]),:);    % A good selective set of configurations

id = temp(:,1);
data = temp(:,2:end);

if(exist('list','var'))
    g = ismember(id,list);
    id = id(g);
    data = data(g,:);
end

% % plot(data');

if(mode==1)
    nBumps = [20 10 6 5 4];             % N bumps per epizode of stimulation
    fitX = [10 20 30 40 50];            % Cos period, ms - to be used as X-coordinates for curve fitting
    dispf('Spiking in resonse to different frequencies');
else
    nBumps = [6 6 6 6 6];
    fitX = [50 90 130 170 210];         % Cosine injection amplitude
    dispf('Spiking in resonse to different amplitudes');
end
nCells = length(id);
if(oneGiantPlot)
    plotN = max(1,floor(sqrt(nCells)/1.3));     % Auto-Layout
    plotM = ceil(nCells/plotN);
else
    plotN = 2;                      % Several pages, but sane size of each fit
    plotM = 3;
end

%%% =================== Spiking as a function of Bump Group (frequency)
if(doGlobal)
    spikePerPeriod  = nan(nCells,length(fitX));	% Average spiking for each period will be stored here
    qBumps = [0 cumsum(nBumps)];                    % Borders of different bump groups
    for(iPeriod=1:length(fitX))
        spikePerPeriod(:,iPeriod) = mean(data(:,qBumps(iPeriod)+1:qBumps(iPeriod+1)),2);
    end

    [f,s] = myfittypes('2exp_prime',max(fitX)*3);               % Compared to 2exp it has slightly different boundaries, and different starting points
    xFit = linspace(min(fitX),max(fitX),50);    
    figure;
    fprintf('id\tbestX\tspikes\tTauRise\tTauDecay\n');
    for(iCell = 1:nCells)
        subplot(plotN,plotM,mod(iCell-1,plotN*plotM)+1); hold on;    
        plot(fitX,spikePerPeriod(iCell,:),'b.');    
        [c2,~] = fit(fitX(:),spikePerPeriod(iCell,:)',f,s);    
        b = c2.b;
        c = c2.c;
        maxx = max(0 , b*c*(log(c)-log(b))/(c-b) + c2.a);   % Where 2exp max is located (best ISI)
        maxx = max(min(fitX),min(maxx,max(fitX)));          % Don't let maxx leave the range that was actually tested
        plot(xFit,c2(xFit),'r-');
        plot(maxx,c2(maxx),'r.');
        if(plotN>6)                                         % Compact form
            set(gca,'FontSize',6,'XTick',[]);               % Small label & no X-axis ticks
        end
        % set(gca,'YTick',[]);                              % No Y-axis ticks (I'm not sure if I like it more with or without it)
        title(sprintf('%d: %d',[iCell id(iCell)]));
        hold off;         
        fprintf('%d\t%d\t%6.4f\t%6.2f\t%6.2f\n',[id(iCell) round(maxx) mean(spikePerPeriod(iCell,:)) c2.c c2.b]);
        drawnow;
        if(iCell<nCells)
            if(mod(iCell,plotN*plotM)==0);            % Time to get a new figure window
                figure;
            end
        end
    end
end

%%% =================== Spiking during high-frequency group
if(doInactivation)
    [f,s] = myfittypes('2exp',max(fitX)*3);       % (x-a)*exp(-(x-b)/c)*d+e : inactivation fit
    xFit = linspace(1,nBumps(1),100);
    nB = nBumps(1);                                 % Local variable (number of fast-bumps) for shorter formulas.
    figure;
    dispf('Inactivation during high-frequency oscillations');
    fprintf('id  \tmaxSp\twave\tTauRise\tTauDecay\n');
    for(iCell = 1:nCells)
        subplot(plotN,plotM,mod(iCell-1,plotN*plotM)+1); hold on;
        plot(data(iCell,1:nBumps(1)),'b.');
        [c2,~] = fit((1:nB)',[data(iCell,1:nB)]',f,s);                
        % maxSpike  = max(data(iCell,1:nB));          % Maximal spiking achieved
        bestWave  = find(c2(xFit)==max(c2(xFit)));  % Best wave estimation (in terms of causing spiking)
        bestWave = bestWave(1);                     % In case all of them are equal
        bestWave  = max(1,min(bestWave,nB));        % bring within the range of tested data.    
        % inactInd  = (maxSpike-c2.e)/maxSpike;     % Inactivation index: 0 for saturation to plato, 1 for full inactivation. Only works for formulas with an asymptote.
        if(length(unique(data(iCell,1:nB)))==1)     % If all measurements are identical (usually 1 spike/bump)
            tauR = 1;                               % Make tau-rise super-short
            tauD = 200;                             % .. and tau-decay - artificially high and round.
        else
            tauR = c2.c;
            tauD = c2.b;                            % else - normal fit calculation (whatever it is)
        end
        plot(xFit,c2(xFit),'r-');
        set(gca,'FontSize',6);
        xlabel(sprintf('%d: %d',[iCell id(iCell)]));
        hold off;
        fprintf('%d\t%4.1f\t%4.2f\t%4.2f\n',[id(iCell) bestWave tau]);
        drawnow;
        if(iCell<nCells)
            if(mod(iCell,plotN*plotM)==0);            % Time to get a new figure window
                figure;
            end
        end
    end
end

end



function [f,s] = myfittypes(kind,maxX)
switch(kind)
    case 'exp'
        f = fittype('exp((x-a)/b)*c+e','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX   1   0   0],...
                       'Upper',     [maxX    maxX Inf Inf],...
                       'Startpoint',[0      10  1   1]);
    case '-exp'
        f = fittype('exp(-(x-a)/b)*c+e','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX      1       0.1       0],...
                       'Upper',     [maxX       Inf     Inf     Inf],...
                       'Startpoint',[0          10      1       1]);
   case '1-exp'
        f = fittype('(1-exp(-(x-a)/b))*c','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX   1      0   ],...
                       'Upper',     [maxX    maxX   Inf ],...
                       'Startpoint',[0       10     1   ]);
    case '2exp'
        f = fittype('max(0,exp(-(x-a)/b)-exp(-(x-a)/c))*d','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [0  0.1     0.1     0],...
                       'Upper',     [maxX   maxX    maxX    Inf],...
                       'Startpoint',[0     maxX/20    maxX/50  2]); % Some sane-looking curve
    case '2exp_prime'   % Compared to 2exp it has slightly different boundaries, and different starting points
        f = fittype('max(0,exp(-(x-a)/b)-exp(-(x-a)/c))*d','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX  0.1         0.1     0],...
                       'Upper',     [maxX   maxX        maxX    Inf],...
                       'Startpoint',[1      110         8       1]); % Some sane-looking curve
    case 'cutexp'
        f = fittype('max(0,exp((x-a)/b)-e)*c+d','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX   1      0   0   0],...
                       'Upper',     [maxX    maxX   Inf Inf 1],...
                       'Startpoint',[0       10     1   1   0.1]);
    case 'sigmoid'
        f = fittype('c/(1+exp(-(x-a)/b)+d)','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX   1   0   0],...
                       'Upper',     [maxX    maxX Inf Inf],...
                       'Startpoint',[0      10  1   1]);
    case 'sigmoidoid'
        f = fittype('c*exp(x/b1)./(1+exp(-(a-x)/b2))','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX   .1      .1        0   ],...
                       'Upper',     [0     maxX    maxX     Inf ],...
                       'Startpoint',[-1       10     10       1   ]);
    case 'xexp' 
        f = fittype('(x-a)*exp(-(x-b)/c)*d+e','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX   -maxX   1      0       0],...
                       'Upper',     [maxX    maxX    maxX   Inf     Inf],...
                       'Startpoint',[1       -9      2      70      0.01]);
    case 'xpexp' 
        f = fittype('(a*x-b)*exp(-x/c)+e','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [0        0        1      0],...
                       'Upper',     [+Inf     maxX     100    100],...
                       'Startpoint',[0        1        10      0.5]);

end % Switch

end